#!/usr/bin/perl -w

###################################
#            MUDShell 0.1         #
# Copyleft 2000, Henning Strandin #
###################################

### USER SETTINGS ###

# This value should be the name of a file
# in /etc/mdsh/skins/ or $HOME/mdsh/skins
my $skin = 'default';

# After how many passive seconds should a user
# be marked as asleep?
my $sleep_limit = 3600;

#####################


use strict;

use Term::ANSIColor qw(:constants);
$Term::ANSIColor::AUTORESET = 1;            

# 'Touch' the user's location file, to update 'last active'
# timestamp.
if (-s "/var/mdsh/where/$ENV{'USER'}"){
  my $now = time;
  utime($now, $now, "/var/mdsh/where/$ENV{'USER'}");
}

# Get command and arguments provided as arguments from
# the shell function, and call the correct procedure.
{
  my ($command, @args) = @ARGV;
  go()                 if $command eq 'go';
  take(@args)          if $command eq 'take';
  drop(@args)          if $command eq 'drop';
  inventory()          if $command eq 'inventory';
  examine(@args)       if $command eq 'examine';
  use_object($args[0]) if $command eq 'use';
  quit()               if $command eq 'quit';
}

# This procedure gets called after the shell function
# has done the 'cd' (i.e. changed the CWD).
sub go {

  # Save the user's current location.
  # This file is also used to deduce how long since
  # it was that the user was active.
  open (SAVE_LOCATION, ">/var/mdsh/where/$ENV{'USER'}");
  print SAVE_LOCATION $ENV{'PWD'};
  close SAVE_LOCATION;

  # Get the people, files, and exits in this directory.
  my %people_here = get_people($ENV{'PWD'});
  my @files = get_dir_contents('files', $ENV{'PWD'});
  my @dirs = get_dir_contents('dirs', $ENV{'PWD'});

  # Print current location.
  print "\n";
  print "Location: ";
  print RED "$ENV{'PWD'}\n";

  # Print a room description if there is one.
  if (-s "$ENV{'PWD'}/.room_description"){
    open (ROOM_DESCRIPTION, "$ENV{'PWD'}/.room_description");
    print "\n";
    while (<ROOM_DESCRIPTION>){
      print RED $_;
    }
    close ROOM_DESCRIPTION;
    print "\n";
  }

  # If no room description is found, look for
  # a description of this room in the current
  # skin file. First look for the skin file
  # in the local directory, then in the global.
  elsif (-s "$ENV{'HOME'}/mdsh/skins/$skin" || -s "/etc/mdsh/skins/$skin"){
    my $current_room = '';
    my $first_empty_row = 0;
    my $skin_file = -s "$ENV{'HOME'}/mdsh/skins/$skin" ?
      "$ENV{'HOME'}/mdsh/skins/$skin" :
	"/etc/mdsh/skins/$skin";
    open (SKIN, $skin_file);
    while (<SKIN>){
      if (/^\//){
	chomp $_;
	$current_room = $_;
	next;
      }
      if ($current_room eq $ENV{'PWD'}){
	unless ($first_empty_row){
	  print "\n";
	  $first_empty_row = 1;
	}
	print RED $_;
      }
    }
    close SKIN;
    print "\n";
  }
  else { print "\n"; }
  
  # Print the number of files in this directory.
  if (@files){
    print "This room contains ";
    print MAGENTA scalar @files;
    if (@files == 1){
      print " file.\n";
    }
    else {
      print " files.\n";
    }    
    print "\n";
  }

  # Print the names of all people in this
  # directory, and report them as sleeping
  # if they haven't been active during the
  # last hour.
  if (%people_here){
    print "Users in this room:\n";
    foreach (keys %people_here){
      print GREEN $_;
      print time - $people_here{$_} > $sleep_limit ? ", sleeping...\n" : "\n";
    }
    print "\n";
  }

  # Print all the directories (exits)
  # availible from this directory (room).
  my $sep;
  print "Exits are:\n";
  for (0 .. @dirs - 1){
    print BLUE "$dirs[$_]";
    if (($_ + 1) % 3 != 0){
      $sep = " " x (20 - length($dirs[$_]));
    }
    else {
      $sep = "\n";
    }
    print $sep;
  }
  print "\n";
}


# Pick up one or more files. Does this by mv'ing them
# to $HOME/.mdsh/inventory.
sub take {
  my $extension = '';
  my @items = @_;
  foreach (@items){

    # No such file.
    unless (-f){
      print "There's no \"$_\" here!\"\n";
      next;
    }

    # Don't have write permisions (so, can't mv).
    unless (-w){
      print "You can't pick up \"$_\", it seems to be stuck to the ground!\n";
      print "Maybe you don't own it..?\n";
      next;
    }

    # If there's already a file by this name, add the next serial
    # number to the end of it.
    if (-s "$ENV{'HOME'}/.mdsh/inventory/$_"){
      $extension = "." . get_next_serial_no($_);
    }

    # Finally, move the file and report it.
    `mv $_ $ENV{'HOME'}/.mdsh/inventory/$_$extension`; # No error checking!!
    print "Took ";
    print MAGENTA "$_\n";
  }
}


# Returns the next availible serial number for files
# with the same name in the inventory.
sub get_next_serial_no {
  my $highest_no = 0;
  my $item = shift;
  foreach (grep /^$item\.\d+$/, get_dir_contents('files', "$ENV{'HOME'}/.mdsh/inventory")){
    my ($serial_no) = /\.(\d+)$/;
    $highest_no = $serial_no > $highest_no ? $serial_no : $highest_no;
  }
  return $highest_no + 1;
}  


# Drop one or more files. Only takes one argument, a regexp.
sub drop {
  my $item = shift;
  my @inventory = get_dir_contents('files', "$ENV{'HOME'}/.mdsh/inventory");
  my @to_drop = grep /$item/, @inventory;

  # No file names in $HOME/.mdsh/inventory matches the given
  # expression.
  unless (@to_drop){
    print "You're not carrying anything matching \"$item\"!\n";
    return;
  }

  # Don't have write permision in CWD (so, can't mv here).
  unless (-w $ENV{'PWD'}){
    print "Oops, you're not allowed to drop anything here.\n";
    return;
  }

  # Drop everything that matches and report it.
  foreach (@to_drop){
    `mv $ENV{'HOME'}/.mdsh/inventory/$_ .`; # No error checking!!
    print "Dropped ";
    print MAGENTA "$_\n";
  }
}


# Print the contents of the inventory.
sub inventory {
  my @items = get_dir_contents('files', "$ENV{'HOME'}/.mdsh/inventory");
  if (@items){
    print "You're carrying:\n";
    print MAGENTA "$_\n" foreach (@items);
  }
  else{
    print "You're not carrying anything.\n";
  }
}


# Examine an object in the current directory.
sub examine {
  my $object = shift || '';
  $object = 'north' if $object eq 'n';
  unless ($object){
    go();
    return;
  }
  $object =~ s/\/$//;
  my $meta_quoted_object = quotemeta($object);
  my %people = get_people($ENV{'PWD'});
  my @files = get_dir_contents('files', $ENV{'PWD'});
  my @dirs = get_dir_contents('dirs', $ENV{'PWD'});

  # Examine a person.
  # Prints the user's description entry in
  # /etc/passwd, and when the user was last
  # active, according to the modification
  # time on the location file.
  if (grep (/^$meta_quoted_object$/, keys %people)){
    open (PASSWD, "/etc/passwd");
    while (<PASSWD>){
      my ($user_name, $user_info) = ((split(/:/, $_))[0], (split(/:/, $_))[4]);
      $user_info or $user_info = "a mystery.";
       if ($user_name eq $object){
	 print GREEN $object;
	 print " is $user_info.\n";
	 print GREEN $object;
	 print " has been passive since " . localtime($people{$object});
	 print ".\n";
       }
    }
    close PASSWD;
  }

  # Examine a file. Runs the 'file' command on the file
  # And prints its output slightly reformatted.
  elsif (grep(/^$meta_quoted_object$/, @files)){
    if (-r $object){
      my $file_output = `file $object`;
      my ($file_name, $file_info) = split(/:/, $file_output);
      $file_name =~ s/^\s+//;
      $file_info =~ s/^\s+//;
      $file_name =~ s/\s+$//;
      $file_info =~ s/\s+$//;
      print "A closer look at ";
      print MAGENTA $file_name;
      print " reveals that it is a $file_info file.\n";
    }
    else {
      print "You can't examine \"$object,\" its owner doesn't allow you.\n";
    }
  }

  # Examine a directory. Reports the number of
  # people, files and exits (directories) in the
  # directory examined.
  elsif (grep(/^$meta_quoted_object$/, @dirs)){

    # Check that the directory is browsable.
    if (-x $object || $object eq 'north'){

      # Get all the info about this directory and print it.
      my %people_in_dir = get_people($object);
      my @files_in_dir = get_dir_contents('files', $object);
      my @dirs_in_dir = get_dir_contents('dirs', $object);
      print BLUE $object;
      print " is a room, with ";
      print GREEN scalar keys %people_in_dir;
      print GREEN scalar keys %people_in_dir == 1 ? " person" : " persons";
      print ", ";
      print MAGENTA scalar @files_in_dir;
      print MAGENTA scalar @files_in_dir == 1 ?	" file " : " files";
      print " and ";
      print BLUE scalar @dirs_in_dir;
      print BLUE scalar @dirs_in_dir == 1 ? " exit" : " exits";
      print ".\n";
    }

    # Directory is not browsable.
    else {
      print "You can't peep through the door to \"$object.\" It won't open.\n";
    }
  }

  # No such object in the CWD.
  else {
    print "There is no $object here!\n";
  }
}


# Takes a directory name, and returns a hash with the names of people
# in the directory as keys and the time (in epocs) of their last
# action as the value. Gets this info from the persons' location
# files, in /var/mdsh/where.
sub get_people {
  my $dir = shift;

  # Turn into a full path if necessary.
  $dir = "$ENV{'PWD'}/$dir" unless $dir =~ /^\// || $dir eq 'north';

  # If the name of the directory is 'north,' remove the last
  # directory from the (full path) CWD and use that.
  ($dir) = $ENV{'PWD'} =~ /(.*)\/[\.\w\+-]+$/ if $dir eq 'north';

  my %people_here;
  opendir (WHERE, "/var/mdsh/where");
  my @people_where = readdir WHERE;
  closedir WHERE;
  foreach (sort @people_where){
    next if $_ eq $ENV{'USER'} || /^\.\.?/;
    open (LOCATION, "/var/mdsh/where/$_");
    my $location = <LOCATION>;
    if ($location eq $dir){
      $people_here{$_} = (stat("/var/mdsh/where/$_"))[9];
    }
  }
  return %people_here;
}


# Takes a type (dirs|files|all) and a directory, and
# returns the contents of that type in that directory.
sub get_dir_contents {
  my ($type, $dir) = @_;
  $dir = '..' if $dir eq 'north';
  $dir = "$ENV{'PWD'}/$dir" unless $dir =~ /^\// || $dir eq 'north';
  my @contents;
  opendir (HERE, $dir);
  my @dirs_n_files = readdir HERE;
  closedir HERE;
  foreach (sort @dirs_n_files){
    if ($type eq 'dirs'){
      $contents[@contents] = $_ if -d "$dir/$_" && !/^\.\.?$/;
    }
    elsif ($type eq 'files'){
      $contents[@contents] = $_ unless -d "$dir/$_";
    }
    elsif ($type eq 'all'){
      $contents[@contents] = $_;
    }
  }      
  $contents[@contents] = 'north' if $dir ne "/" && $type eq 'dirs';
  return @contents;
}


# Takes a file name and, if it's an executable, executes it and exits.
# If it's not, using the 'file' command, it tries to match its type to an
# entry in mdsh.magic and open it with an apropriate application.
sub use_object {
  my $object = shift;
  unless ($object){
    print "Use what?\n";
    return;
  }

  # The argument isn't a file.
  unless (-f $object){
    print "There's no ";
    print MAGENTA "\"$object\"";
    print " here...\n";
    return;
  }

  # The argument is an executable. Execute it.
  if (-x $object){
    print "\"$object\" is apparently a magical object! You hear a faint\n";
    print "rustling and the air fills with little multi-colored sparks.\n";
    print "The magical charge makes the hair on your arms stand up...\n";
    print "\n";
    print "Press Enter to continue...\n";
    my $pause = <STDIN>;

    # If the program name begins with 'x' we fork and launch.
    if ($object =~ /^x/){
      my $pid = fork();
      exec $object if $pid == 0;
      exit;
    }
    else {
      exec $object;
    }
  }

  # The argument is not an executable. Try to open it with an apropriate
  # aplication, if we have read permission.
  if (-r $object){
    my $file_type = `file $object`;
    open (MAGIC, "$ENV{'HOME'}/mdsh/mdsh.magic");
    my @file_defs = <MAGIC>;
    close MAGIC;
    chomp(@file_defs);
    
    # For each type definition in mdsh.magic, look if it matches the
    # name of the files that we try to use. If it does, check the $PATH
    # to see if an apropriate application is availible. The first one
    # we find, we execute with the file as an argument, and exits.
    foreach(@file_defs){
      my ($all_type_defs, $all_app_defs) = split(/\t/);
      my @type_defs = split(/,/, $all_type_defs);
      my @app_defs = split(/,/, $all_app_defs);
      foreach my $type_def (@type_defs){
	if ($file_type =~ /$type_def/){
	  foreach my $app_and_fork_info (@app_defs){
	    my $fork = 1 if $app_and_fork_info =~ / &$/;
	    my ($app) = $app_and_fork_info =~ /([^ &]*)/;
	    foreach my $env_path (split(/:/, $ENV{'PATH'})){
	      if (-x "$env_path/$app"){
		
		# If the application name has a trailing ' &' in
		# mdsh.magic, fork and launch.
		if ($fork){
		  my $pid = fork();
		  exec "$env_path/$app", $object if $pid == 0;
		  exit;
		}
		else {
		  exec "$env_path/$app", $object;
		}
	      }
	    }
	  }
	}
      }
    }
  }

  # If the file isn't readable.
  else {
    print "You're not allowed to use this object.\n";
    return;
  }
  
  # If no matching file type is found, or none of the apropriate
  # applications named in mdsh.magic are availible.
  print "You don't seem to know how to use \"$object\"...\n";
}


# Die.
sub quit {
  unlink "/var/mdsh/where/$ENV{'USER'}";
  print RED "You feel a sudden sharp pain in your chest,\n";
  print RED "and everything... goes black.\n";
  print "\n";
  print "Thank you for playing MUDShell.\n";
  print "Press Enter to quit.\n";
  my $pause = <STDIN>;
}
